// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


package com.vasco.digipass.sdk.utils.notification.client.sample;

import android.Manifest;
import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.graphics.Insets;
import androidx.core.view.OnApplyWindowInsetsListener;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.core.view.WindowInsetsControllerCompat;

import com.vasco.digipass.sdk.utils.notification.client.NotificationSDKClient;
import com.vasco.digipass.sdk.utils.notification.client.constants.NotificationSDKClientErrorCodes;
import com.vasco.digipass.sdk.utils.notification.client.exceptions.NotificationSDKClientException;

import java.util.Map;
import java.util.Objects;

/**
 * Test class for the Notification SDK Client. It demonstrates how to:
 *
 * <ul>
 *   <li>register your device to the notification platform
 *   <li>retrieve the VASCONotificationIdentifier that must be provided to the NotificationSDKServe
 *       Sample
 *   <li>receive a notification, and retrieve its content.
 * </ul>
 */
public class NotificationSDKClientSampleActivity extends AppCompatActivity {
    private static final String TAG = NotificationSDKClientSampleActivity.class.getName();
    TextView errorTextView;
    TextView vascoNotificationIdentifierTextView;
    TextView notificationContentTextView;
    TextView notificationCustomAttributesTextView;

    private static final int REQUEST_PERM_CODE = 1515;

    protected void registerVascoSDK() {
        final NotificationSDKClient.NotificationSDKClientListener listener =
                new NotificationSDKClient.NotificationSDKClientListener() {

                    public void onRegistrationSuccess(String vascoNotificationIdentifier) {
                        runOnUiThread(
                                () -> {
                                    Log.i(TAG, "VASCO notification identifier:");
                                    Log.i(TAG, vascoNotificationIdentifier);

                                    // display VASCO Notification Identifier on screen
                                    vascoNotificationIdentifierTextView.setText(
                                            vascoNotificationIdentifier);
                                });

                        // copy VASCO Notification Identifier into clipboard
                        ClipboardManager clipboard =
                                (ClipboardManager) getSystemService(CLIPBOARD_SERVICE);
                        ClipData clipData =
                                ClipData.newPlainText(
                                        "VASCO Notification Identifier",
                                        vascoNotificationIdentifier);
                        Objects.requireNonNull(clipboard).setPrimaryClip(clipData);
                    }

                    public void onException(NotificationSDKClientException e) {
                        final String errorMsg =
                                "NotificationSDKClientException error code: " + e.getErrorCode();
                        Log.e(TAG, errorMsg);
                        runOnUiThread(
                                () -> {
                                    errorTextView.setText(errorMsg);
                                    vascoNotificationIdentifierTextView.setText("");
                                    if (e.getErrorCode()
                                            == NotificationSDKClientErrorCodes
                                                    .PERMISSION_NOT_GRANTED) {
                                        requestPermission();
                                    }
                                });
                    }
                };

        // Register to Firebase Cloud Messaging
        // and retrieving the VASCO Notification Identifier that must be provided
        // to the NotificationSDKServer library to send a notification to this particular device.
        NotificationSDKClient.registerNotificationService(this, listener);
    }

    private void requestPermission() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            requestPermissions(
                    new String[] {Manifest.permission.POST_NOTIFICATIONS}, REQUEST_PERM_CODE);
        }
    }

    @Override
    public void onRequestPermissionsResult(
            int requestCode, String[] permissions, int[] grantResults) {
        if (requestCode == REQUEST_PERM_CODE) {
            if (grantResults.length <= 0 || grantResults[0] != PackageManager.PERMISSION_GRANTED) {
                Toast.makeText(this, R.string.required_permission_needed, Toast.LENGTH_SHORT)
                        .show();
            }
            recreate();
        }
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

        errorTextView = findViewById(R.id.errorTextView);
        vascoNotificationIdentifierTextView =
                findViewById(R.id.vascoNotificationIdentifierTextView);
        registerForContextMenu(vascoNotificationIdentifierTextView);
        notificationContentTextView = findViewById(R.id.notificationTextView);
        notificationCustomAttributesTextView =
                findViewById(R.id.notificationCustomAttributesTextView);

        registerVascoSDK();

        // Display notification content if the intent is a notification
        displayNotificationContent(getIntent());

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            setInsets(getWindow(), findViewById(R.id.scroll));
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.Q)
    public void setInsets(Window window, View rootView) {
        WindowInsetsControllerCompat controller =
                new WindowInsetsControllerCompat(window, window.getDecorView());
        controller.setAppearanceLightStatusBars(true);
        ViewCompat.setOnApplyWindowInsetsListener(rootView, new OnApplyWindowInsetsListener() {
            @NonNull
            @Override
            public WindowInsetsCompat onApplyWindowInsets(@NonNull View v, @NonNull WindowInsetsCompat insets) {
                int systemBars = WindowInsetsCompat.Type.systemBars();
                Insets systemBarInsets = insets.getInsets(systemBars);

                ViewGroup.MarginLayoutParams layoutParams = (ViewGroup.MarginLayoutParams) v.getLayoutParams();
                layoutParams.topMargin = systemBarInsets.top;
                layoutParams.bottomMargin = systemBarInsets.bottom;
                layoutParams.leftMargin = systemBarInsets.left;
                layoutParams.rightMargin = systemBarInsets.right;

                v.setLayoutParams(layoutParams);
                return WindowInsetsCompat.CONSUMED;
            }
        });
    }

    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);

        // Display notification content if the intent is a notification
        displayNotificationContent(intent);
    }

    private void displayNotificationContent(Intent intent) {
        try {
            // Check if the intent is a notification
            if (NotificationSDKClient.isVASCONotification(intent)) {
                // Parse the notification and retrieve its content.
                String notificationContent = NotificationSDKClient.parseVASCONotification(intent);

                Log.v(TAG, "Notification content:");
                Log.v(TAG, notificationContent);
                Map<String, String> customAttributes =
                        NotificationSDKClient.getCustomAttributes(intent);
                for (Map.Entry<String, String> entry : customAttributes.entrySet()) {
                    Log.v(TAG, "Custom attribute: " + entry.getKey() + " - " + entry.getValue());
                }

                if (NotificationSDKClient.isForegroundNotification(intent)) {
                    // If notificationsOpensInForeground is set to true in
                    // onespan-notifications.properties,
                    // a different behavior can be applied here for Notifications received by the
                    // server
                    // while the application is in foreground
                    // If the feature is disabled, this if/else block is useless.
                    notificationContentTextView.setText(
                            String.format(
                                    "Received Foreground notification:%n%s", notificationContent));
                } else {
                    notificationContentTextView.setText(notificationContent);
                }

                // if custom attributes are given, we display them
                if (!customAttributes.isEmpty()) {
                    StringBuilder customAttributesAsString =
                            new StringBuilder("Custom attributes:\n");
                    for (Map.Entry<String, String> entry : customAttributes.entrySet()) {
                        customAttributesAsString
                                .append(entry.getKey())
                                .append(" - ")
                                .append(entry.getValue())
                                .append("\n");
                    }
                    notificationCustomAttributesTextView.setText(
                            customAttributesAsString.toString());
                }
            }
        } catch (NotificationSDKClientException e) {
            String errorMsg = "NotificationSDKClientException error code: " + e.getErrorCode();

            Log.e(TAG, errorMsg);
            errorTextView.setText(errorMsg);
        }
    }
}
